/*:
 * @target MZ
 * @plugindesc アクターが特定のスキルを習得している場合、ターン終了時に確率で自動発動しメッセージを表示します。
 * @author あなたの名前
 * @help
 * アクターが特定のスキルを習得している場合、
 * 各ターン終了時にそのスキルを自動で発動します（演出付き）。
 * さらに、発動確率を設定でき、発動時にはメッセージが表示されます。
 *
 * 【使い方】
 * 1. 自動発動させたいスキルのメモ欄に以下のいずれかを記述してください。
 *    - `<AutoTurnEndSkill>`: 100%の確率で発動します。
 *    - `<AutoTurnEndSkill:確率>`: 指定した確率（例: `<AutoTurnEndSkill:50>` で50%）で発動します。
 *      確率は0〜100の整数で指定します。
 *
 * 例：
 *   - スキル「自己回復」のメモ欄に `<AutoTurnEndSkill>` と記述すると、
 *     「自己回復」を覚えているアクターがターン終了時に100%の確率で自動でスキルを発動します。
 *   - スキル「状態異常解除」のメモ欄に `<AutoTurnEndSkill:75>` と記述すると、
 *     「状態異常解除」を覚えているアクターがターン終了時に75%の確率で自動でスキルを発動します。
 *
 * 2. 複数のスキルにタグを設定すれば、それらを覚えている各アクターが
 *    それぞれのスキルを確率に応じて発動し、メッセージを表示します。
 *
 * @command dummy
 * @text 設定項目なし
 * @desc このプラグインに設定項目はありません。
 */

(() => {
    const pluginName = "IndividualAutoSkillOnTurnEndWithProb";
    const autoSkillTag = "AutoTurnEndSkill";

    // Game_BattlerBase.prototype.regenerateAll を参考に、
    // ターン終了時の自動処理に割り込みます。
    const _BattleManager_endTurn = BattleManager.endTurn;
    BattleManager.endTurn = function() {
        this._phase = 'auto_turn_end_skill'; // フェーズを一時的に設定

        let forcedActions = []; // 強制実行するアクションを一時的に保持する配列

        // 全てのパーティメンバーと敵キャラクターを対象にチェック
        const allBattlers = $gameParty.battleMembers().concat($gameTroop.members());
        
        allBattlers.forEach(battler => {
            if (battler.isAlive()) { // 生存しているキャラクターのみ対象
                // アクターの場合は習得スキル、敵の場合は設定されているスキルをチェック
                const skillsToCheck = battler.isActor() ? battler.skills() : battler.enemy().actions.map(a => $dataSkills[a.skillId]).filter(s => s);

                skillsToCheck.forEach(skill => {
                    if (skill && skill.meta[autoSkillTag]) {
                        // メモ欄のタグから確率を解析
                        let activateRate = 100; // デフォルトは100%
                        const tagValue = skill.meta[autoSkillTag];
                        if (typeof tagValue === 'string') {
                            const rateMatch = tagValue.match(/^(\d+)$/);
                            if (rateMatch) {
                                activateRate = Number(rateMatch[1]);
                            }
                        }

                        // 確率判定
                        if (Math.random() * 100 < activateRate) {
                            // 発動メッセージの表示
                            const battlerName = battler.name();
                            const skillName = skill.name;
                            $gameMessage.add(`${battlerName}の${skillName}が発動した！`);

                            // 強制アクションを登録
                            battler.forceAction(skill.id, -1); // ターゲットは自動
                            BattleManager.forceAction(battler);
                        }
                    }
                });
            }
        });

        // 強制されたアクションがあれば実行を開始
        // ターン終了時にまとめて実行されるようにする
        BattleManager.processForcedAction();

        // 強制アクションの処理が完了したら、元のターン終了処理に戻る
        const _Scene_Battle_update = Scene_Battle.prototype.update;
        Scene_Battle.prototype.update = function() {
            _Scene_Battle_update.call(this);
            if (!BattleManager.isBusy() && BattleManager._phase === 'auto_turn_end_skill') {
                Scene_Battle.prototype.update = _Scene_Battle_update; // 元のupdateに戻す
                _BattleManager_endTurn.call(BattleManager); // 元のターン終了処理を呼び出す
            }
        };
    };
})();
